/*
 * \file   main.c
 *
 * \brief  Main routine for the bootloader for G2553
 *
 */
/* --COPYRIGHT--,BSD
 * Copyright (c) 2017, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
//
//  Include files
//
#include <stdbool.h>
#include <msp430x471x7.h>
#include "TI_MSPBoot_Common.h"
#include "TI_MSPBoot_CI.h"
#include "TI_MSPBoot_AppMgr.h"
#include "TI_MSPBoot_CI_PHYDL.h"

//
//  Local function prototypes
//
inline static void Clock_init(void);
inline static void HW_init(void);
void FLL_Init_MF(void);
/******************************************************************************
 *
 * @brief   Main function
 *  - Initializes the MCU
 *  - Selects whether to run application or bootloader
 *  - If bootloader:
 *      - Initializes the peripheral interface
 *      - Waits for a command
 *      - Sends the corresponding response
 *  - If application:
 *      - Jump to application
 *
 *  @note   USI interrupts are enabled after this function
 * @return  none
 *****************************************************************************/
int main_boot( void )
{
    // Stop watchdog timer to prevent time out reset
    WDTCTL = WDTPW + WDTHOLD;

   /* 
    * Important Note: 
    *   MSPBoot assumes that proper conditions are met when executing after reset. 
    *   Its important to remark that DCO must not be changed until VCC>=VCC(min), 
    *   and a minimum of 2.2V is required for Flash Program/Erase operations.
    *   An optional software delay can be added for applications having a slow 
    *   voltage ramp-up or an external voltage supervisor is suggested
    *   
    */
    //  __delay_cycles(10000);  // Uncomment this line and adjust number of delay 
                                // cycles according to the application voltage ramp
    // Initialize MCU
    HW_init();

    Clock_init();

        P9DIR |= BIT3;
        P9OUT |= BIT3;
        __delay_cycles(500000);
        P9OUT &= ~BIT3;
        __delay_cycles(500000);
        P9OUT |= BIT3;
        __delay_cycles(500000);
        P9OUT &= ~BIT3;


    // Validate the application and jump if needed
    if (TI_MSPBoot_AppMgr_ValidateApp() == true)
        TI_MSPBoot_APPMGR_JUMPTOAPP();

    TI_MSPBoot_CI_Init();      // Initialize the Communication Interface

#ifndef NDEBUG
    P1DIR |= BIT0 + BIT6;     // Used for debugging purposes to show entry to MSPBoot
    P1OUT |= BIT0 + BIT6;
#endif

    while(1)
    {
        // Poll PHY and Data Link interface for new packets
        TI_MSPBoot_CI_PHYDL_Poll();
               __delay_cycles(500000);
               __delay_cycles(500000);
               __delay_cycles(500000);
                P9OUT |= BIT3;
               __delay_cycles(500000);
               __delay_cycles(500000);
               __delay_cycles(500000);
               P9OUT &= ~BIT3;

        // If a new packet is detected, process it
        if (TI_MSPBoot_CI_Process() == RET_JUMP_TO_APP)
        {
            // If Packet indicates a jump to App
            TI_MSPBoot_AppMgr_JumpToApp();
        }
#ifdef NDEBUG
        // Feed the dog every (interval depends on MCLK freq)
        WATCHDOG_FEED();
#endif
    }


}


/******************************************************************************
 *
 * @brief   Initializes the MSP430 Clock
 *
 * @return  none
 *****************************************************************************/
inline static void Clock_init(void)
{
#if (MCLK==1000000)
    // Validate the calibration values (only in Debug)
    ASSERT_H((CALBC1_1MHZ !=0xFF) && (CALDCO_1MHZ != 0xFF));

    BCSCTL1 = CALBC1_1MHZ;              // Set DCO to default of 1Mhz
    DCOCTL = CALDCO_1MHZ;
#elif (MCLK==4000000)
    // Validate the calibration values (only in Debug)
    ASSERT_H((CALBC1_8MHZ !=0xFF) && (CALDCO_8MHZ != 0xFF));

    BCSCTL1 = CALBC1_8MHZ;              // Set DCO to default of 8Mhz
    DCOCTL = CALDCO_8MHZ;
    BCSCTL2 = DIVM_1 | DIVS_1;          // MCLK/SMCLK = 8Mhz/2 = 4Mhz
    #elif (MCLK==8000000)
    // Validate the calibration values (only in Debug)
    //ASSERT_H((CALBC1_8MHZ !=0xFF) && (CALDCO_8MHZ != 0xFF));

    //BCSCTL1 = CALBC1_8MHZ;              // Set DCO to default of 8Mhz
    //DCOCTL = CALDCO_8MHZ;
#else
  #error "Please define a valid MCLK or add configuration"
#endif
    //BCSCTL3 = LFXT1S_2;                // LFXT1 = VLO
    //IFG1 &= ~(OFIFG);                   // Clear OSCFault flag
    FLL_Init_MF();
}

void FLL_Init_MF(void)
{
	int i;
   FLL_CTL0 |= DCOPLUS + XCAP11PF;           // DCO+ set, freq = xtal x D x N+1
   SCFI0 = FLLD_4+FN_4;                      // x4 DCO freq, 16MHz nominal DCO
   SCFQCTL = 127;                            // (127+1) x 32768 x 4 = 16.77 MHz
   FLL_CTL1 |= XT2OFF;
  // FLL_State=FLL_MF_MODE;
  do
  {
    IFG1 &= ~OFIFG;                           // Clear OSCFault flag
    for (i = 0x47FF; i > 0; i--);             // Time for flag to set
  }
  while ((IFG1 & OFIFG));
}
/******************************************************************************
 *
 * @brief   Initializes the basic MCU HW
 *
 * @return  none
 *****************************************************************************/
inline static void HW_init(void)
{
    // Just initialize S2 button to force BSL mode
    P1OUT |= BIT3;
    P1REN |= BIT3;
    // The following routines disable interrupts which are cleared by POR but
    // not by a PUC. In G2xx, these are: Timers, CompA, ADC10
#ifdef HW_RESET_PUC
    TA0CTL = 0x00;
    #ifdef __MSP430_HAS_T1A3__
    TA1CTL = 0x00;
    #endif
    #ifdef __MSP430_HAS_CAPLUS__
    CACTL1 = 0x00;
    #endif
    #ifdef __MSP430_HAS_ADC10__
    ADC10CTL0 = 0x00;
    #endif
#endif
}
